# Copyright 2019 Dan Smith <chirp@f.danplanet.com>
# Copyright 2024 weboo
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import argparse
import builtins
import ctypes
import json
import logging
import os
import subprocess
import sys
import time

if sys.version_info < (3, 10):
    import importlib_resources
else:
    import importlib.resources as importlib_resources

from wx.adv import SplashScreen

from chirp import directory
from chirp import logger

LOG = logging.getLogger(__name__)
CONF = None
APP_NAME = 'CHIRP for UV-K5-RX-JP'

def maybe_install_desktop():
    local = os.path.join(os.path.expanduser('~'), '.local')
    desktop_path = os.path.join(local, 'share',
                                'applications', 'chirp.desktop')
    with importlib_resources.as_file(
            importlib_resources.files('chirp.share')
            .joinpath('chirp.desktop')) as desktop_src:
        with open(desktop_src) as f:
            desktop_content = f.readlines()
    with importlib_resources.as_file(
            importlib_resources.files('chirp.share')
            .joinpath('chirp.ico')) as p:
        icon_path = str(p)
    if os.path.exists(desktop_path):
        return

    import wx
    r = wx.MessageBox(
        _('Would you like CHIRP to install a desktop icon for you?'),
        _('Install desktop icon?'), style=wx.YES_NO)
    if r != wx.YES:
        CONF.set_bool('offered_desktop', True, 'state')
        return

    os.makedirs(os.path.dirname(desktop_path), exist_ok=True)

    # Try to run chirp by name from ~/.local/bin
    exec_path = os.path.join(local, 'bin', 'chirp')
    if not os.path.exists(exec_path):
        # If that doesn't work, then just run it with our python via
        # module, which should always work
        exec_path = '%s -mchirp.wxui' % sys.executable
    with open(desktop_path, 'w') as f:
        for line in desktop_content:
            if line.startswith('Exec'):
                f.write('Exec=%s\n' % exec_path)
            elif line.startswith('Icon'):
                f.write('Icon=%s\n' % icon_path)
            else:
                f.write(line)
        LOG.debug('Wrote %s with exec=%r icon=%r' % (
            f.name, exec_path, icon_path))


def show_error_modal(wx, msg):
    dlg = wx.MessageDialog(None, msg, 'エラー', wx.OK | wx.ICON_ERROR)
    dlg.ShowModal()
    dlg.Destroy()


def chirpmain():
    global CONF
    import wx
    # This must be imported before wx.App() to squelch warnings on startup
    # about duplicate "Windows bitmap file" handlers
    import wx.richtext

    from chirp.wxui import config
    CONF = config.get()

    actions = ['upload', 'download', 'query_rr', 'query_mg',
               'query_rb', 'query_dm', 'new']

    parser = argparse.ArgumentParser()
    parser.add_argument("files", metavar="file", nargs='*',
                        help="File to open")
    parser.add_argument("--module", metavar="module",
                        help="Load module on startup")
    logger.add_version_argument(parser)
    parser.add_argument("--profile", action="store_true",
                        help="Enable profiling")
    parser.add_argument("--onlydriver", nargs="+",
                        help="Include this driver while loading")
    parser.add_argument("--inspect", action="store_true",
                        help="Show wxPython inspector")
    parser.add_argument('--page', default=None,
                        help='Select this page of the default editor at start')
    parser.add_argument('--action', default=None, choices=actions,
                        help='Start UI action immediately')
    parser.add_argument('--restore', default=None, action='store_true',
                        help="Restore previous tabs")
    parser.add_argument('--force-language', default=None,
                        help='Force locale to this ISO language code')
    if sys.platform == 'linux':
        parser.add_argument('--no-linux-gdk-backend', action='store_true',
                            help='Do not force GDK_BACKEND=x11')
        parser.add_argument('--install-desktop-app', action='store_true',
                            default=not CONF.get_bool('offered_desktop',
                                                      'state'),
                            help='Prompt to install a desktop icon')
    logger.add_arguments(parser)
    args = parser.parse_args()
    logger.handle_options(args)

    # support for high DPI display on Windows
    if sys.platform == 'win32':
        try:
            ctypes.windll.shcore.SetProcessDpiAwareness(2)
        except:
            LOG.info('Failed to apply DpiAwareness')

    app = wx.App()
    instance = wx.SingleInstanceChecker(f'{APP_NAME}-{wx.GetUserId()}')
    if instance.IsAnotherRunning():
        dialog = wx.MessageDialog(None, 'アプリケーションはすでに起動中です。', APP_NAME, wx.ICON_ERROR)
        dialog.ShowModal()
        dialog.Destroy()
        wx.Exit()

    splash = RXJPSplashScreen()
    splash.CenterOnScreen(wx.BOTH)
    splash.Show(True)

    app.home_url = app.api_url = app.api_token = ''
    cmd = ["chirp_agent.exe"]

    try:
        result = subprocess.run(cmd, capture_output=True, check=False, text=True, encoding='UTF-8')
        if result.returncode != 0:
            errmsg = json.loads(result.stdout)
            show_error_modal(wx, errmsg['message'])
            sys.exit()

        params = json.loads(result.stdout)
        app.home_url = params['home_url']
        app.api_url = params['api_url']
        app.api_token = params['token']
    except FileNotFoundError:
        pass
    except Exception as e:
        LOG.error(f'check license error')
        LOG.exception(e)
        show_error_modal(wx, 'ライセンス情報の確認に失敗しました。インターネット接続を確認してください。\n無線機との接続が制限されます。')

    if args.force_language:
        force_lang = wx.Locale.FindLanguageInfo(args.force_language)
        if force_lang is None:
            print('Failed to find language %r' % args.force_language)
            return 1
        LOG.info('Forcing locale to %r (%s)' % (
            args.force_language, force_lang.Description))
        lang = force_lang.Language
    elif CONF.is_defined('force_language', 'prefs'):
        prefs_lang = CONF.get('force_language', 'prefs')
        force_lang = wx.Locale.FindLanguageInfo(prefs_lang)
        if force_lang is None:
            LOG.warning('Config prefs.force_language specifies unknown '
                        'language %r', prefs_lang)
            lang = wx.Locale.GetSystemLanguage()
        else:
            LOG.info('Forcing locale to %r (%s) via config', prefs_lang,
                     force_lang.Description)
            lang = force_lang.Language
    else:
        lang = wx.Locale.GetSystemLanguage()

    localedir = str(os.path.join(importlib_resources.files('chirp'),
                                 'locale'))
    app._lc = wx.Locale()
    if localedir and os.path.isdir(localedir):
        wx.Locale.AddCatalogLookupPathPrefix(localedir)
    else:
        LOG.warning('Did not find localedir: %s' % localedir)

    if lang != wx.LANGUAGE_UNKNOWN:
        app._lc.Init(lang)
    else:
        LOG.warning('Initializing locale without known language')
        app._lc.Init()
    app._lc.AddCatalog('CHIRP')
    builtins._ = wx.GetTranslation
    builtins.ngettext = wx.GetTranslation
    LOG.debug('Using locale: %s (%i)',
              app._lc.GetCanonicalName(),
              lang)
    LOG.debug('Translation loaded=%s for CHIRP: %s (%s) from %s',
              app._lc.IsLoaded('CHIRP'),
              wx.Translations.Get().GetBestTranslation('CHIRP'),
              ','.join(
                  wx.Translations.Get().GetAvailableTranslations('CHIRP')),
              localedir)
    LOG.debug('Translation loaded=%s for wxstd: %s (%s)',
              app._lc.IsLoaded('wxstd'),
              wx.Translations.Get().GetBestTranslation('wxstd'),
              ','.join(
                  wx.Translations.Get().GetAvailableTranslations('wxstd')))

    from chirp.wxui import main
    from chirp.wxui import report

    logging.getLogger('main').info(report.get_environment())

    directory.import_drivers(limit=args.onlydriver)

    if CONF.get_bool('developer', 'state'):
        from chirp.drivers import fake
        fake.register_fakes()

    # wxGTK on Wayland seems to have problems. Override GDK_BACKEND to
    # use X11, unless we were asked not to
    if sys.platform == 'linux' and not args.no_linux_gdk_backend:
        os.putenv('GDK_BACKEND', 'x11')

    app.SetAppName('CHIRP')
    mainwindow = main.ChirpMain(None, title='CHIRP')
    mainwindow.Show()

    if args.module:
        mainwindow.load_module(args.module)

    if args.restore or CONF.get_bool('restore_tabs', 'prefs'):
        restored = mainwindow.restore_tabs(None)
    else:
        restored = []

    for fn in args.files:
        if os.path.abspath(fn) in restored:
            LOG.info('File %s on the command line is already being restored',
                     fn)
            continue
        mainwindow.open_file(fn, select=False)

    if args.page:
        mainwindow.current_editorset.select_editor(name=args.page)

    if args.inspect:
        from wx.lib import inspection
        inspection.InspectionTool().Show()

    if args.action:
        wx.CallAfter(getattr(mainwindow, '_menu_%s' % args.action), None)

    report.check_for_updates(
        lambda ver: wx.CallAfter(main.display_update_notice, ver))

    if sys.platform == 'linux' and args.install_desktop_app:
        try:
            maybe_install_desktop()
        except Exception as e:
            LOG.exception('Failed to run linux desktop installer: %s', e)

    app.MainLoop()


class RXJPSplashScreen(SplashScreen):
    def __init__(self, parent=None):
        import wx

        with importlib_resources.as_file(
            importlib_resources.files('chirp.share')
            .joinpath('splash111.jpg')
        ) as splash:
            image = wx.Image(str(splash), wx.BITMAP_TYPE_JPEG)

        frame = wx.Frame(None)
        dpi_scale = frame.GetDPIScaleFactor()
        frame.Destroy()

        original_width, original_height = image.GetSize()
        new_width = int(original_width * dpi_scale / 2)
        new_height = int(original_height * dpi_scale / 2)
        image = image.Scale(new_width, new_height, wx.IMAGE_QUALITY_BICUBIC)

        super(RXJPSplashScreen, self).__init__(
            bitmap=wx.Bitmap(image),
            splashStyle=wx.adv.SPLASH_CENTRE_ON_SCREEN | wx.adv.SPLASH_TIMEOUT,
            milliseconds=1000,
            parent=None,
            id=-1,
            style=wx.STAY_ON_TOP | wx.BORDER_NONE
        )
        self.Bind(wx.EVT_CLOSE, self.OnExit)

    def OnExit(self, event):
        event.Skip()
        self.Hide()
